﻿/*******************************************************************************************************************************************************
*    All Project Files Copyright © 2025 by The ep5 Educational Broadcasting Foundation                                                                 *
*                                                                                                                                                      *
*    Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:    *
*                                                                                                                                                      *
*        →  Redistributions of source code must retain the above copyright notice, this list of conditions, and the following disclaimer:              *
*        →  Redistributions in binary form must reproduce the above copyright notice, this list of conditions, and the following disclaimer in the     *
*           documentation and/or other materials provided with the distribution.                                                                       *
*        →  Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived from this    *
*           software without specific prior written permission.                                                                                        *
*                                                                                                                                                      *
*    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS “AS IS” AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED     *
*    TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO  EVENT SHALL THE COPYRIGHT HOLDER OR     *
*    CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,         *
*    PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF         *
*    LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT, INCLUDING NEGLIGENCE OR OTHERWISE, ARISING IN ANY WAY OUT OF THE USE OF THIS           *
*    SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                                                                      *
*******************************************************************************************************************************************************/

//  Written/updated 27 July 2025 by M Harb and David Fisher
//  Copyright © 2025 by The ep5 Educational Broadcasting Foundation; all rights reserved

//  This demo connects to a Protos X PX-TCP1 fieldbus coupler with installed digital and analogue I/O terminals

using Modbus.Device;                                                 // hard dependency on Nmodbus4
using System.Net.Sockets;
using static System.Console;

namespace ProtosXdigitalDemo
{
    public class Program
    {
        static readonly string ipAddress = "10.10.1.1";              // replace with the device's IP address
        static readonly int port = 502;                              // default Modbus TCP port
        static readonly ushort coilInputStartAddress = 0;            // starting address for digital inputs
        static readonly ushort coilOutputStartAddress = 0;           // starting address for writing to digital outputs
        static string? recipeName;                                   // name of recipe data file in SQL database
        static int timesToRepeat;                                    // 0 = run once only (default); >0 = number of repeat invocations
        static ushort numberOfDigitalInputs;                         // number of discrete inputs to read
        static ushort numberOfDigitalOutputs;                        // number of discrete outputs to write
        static ushort numberOfAnalogueInputs;                        // number of analogue inputs to read
        static ushort numberOfAnalogueOutputs;                       // number of analogue outputs to write

        public static int Main(string[] args)
        {
            WindowUtility.SetAppearanceOptions();                    // set console size
            WindowUtility.MoveWindowToCenter();                      // set console window in center of display
            BackgroundColor = ConsoleColor.DarkBlue;                 // add a bit of color
            ForegroundColor = ConsoleColor.White;
            Clear();                                                 // start with tabula rasa
            switch (args.Length)
            {
                case 0:
                    recipeName = null;
                    timesToRepeat = 0;
                    break;
                case 1:
                    recipeName = args[0];                            // name of recipe to run
                    timesToRepeat = 0;
                    break;
                case 2:
                    recipeName = args[0];                            // name of recipe to run
                    timesToRepeat = Convert.ToInt16(args[1]);        // number of repeats
                    break;
                default:
                    WriteLine("Too many command-line arguments; program fails here");
                    return Data.MachineState.failureCode[2];
            }     
            LoadFailureCodes.FailureCodeValues();                    // seed for much more comprehensive error handler
            using TcpClient client = CreateTcpClient(ipAddress, port);
            var modbusMaster = ModbusIpMaster.CreateIp(client);
            numberOfDigitalInputs = (ushort)Data.MachineState.digitalInputChannels.Length;
            numberOfDigitalOutputs = (ushort)Data.MachineState.digitalOutputChannels.Length;
            WriteLine("Created Modbus master.");
            ReadDiscreteInputs(modbusMaster);
            WriteLine("All digital inputs have been read");

            //  begin test of machine-state data structure
            WriteLine("\nTurning all the outputs OFF...");
            for (byte indx = 0; indx < 16; indx++)                   // set machine state values to all outputs OFF
            {                                                        // this is internal to business rules processing
                Data.MachineState.digitalOutputChannels[indx] = false;
            }
            if (WriteMachineDigitalStateToIO.WriteMachineStateToIO(modbusMaster, coilOutputStartAddress) == Data.MachineState.failureCode[0])      // write the machine state values to the I/O hardware
            {
                WriteLine("All digital outputs have been turned OFF");
                WriteLine("Press the <ANY> key to test the write outputs function...");
                _ = ReadKey(true);
            }
            else
            {
                WriteLine("Program fails here...");
                return Data.MachineState.failureCode[1];
            }
           
            WriteLine("Turning all the outputs ON...");
            for (byte indx = 0; indx < 16; indx++)                   // set machine state values to all outputs ON
            {
                Data.MachineState.digitalOutputChannels[indx] = true;
            }
            if (WriteMachineDigitalStateToIO.WriteMachineStateToIO(modbusMaster, coilOutputStartAddress) == Data.MachineState.failureCode[0])      // write the machine state values to the I/O hardware
            {
                WriteLine("All digital outputs are now ON");
                WriteLine("Press the <ANY> key to turn all outputs OFF and end the program");
                _ = ReadKey(true);
            }
            else
            {
                WriteLine("Writing machine-state variable failed...press the <ANY> key to end the program");
                _ = ReadKey(true);
                return Data.MachineState.failureCode[1];
            }
            WriteLine("Turning all the outputs OFF...");
            for (byte indx = 0; indx < 16; indx++)                   // set machine state values to all outputs OFF
            {                                                        // this is internal to business rules processing
                Data.MachineState.digitalOutputChannels[indx] = false;
            }
            _ = WriteMachineDigitalStateToIO.WriteMachineStateToIO(modbusMaster, coilOutputStartAddress);                 // write the machine state values to the I/O hardware
            return Data.MachineState.failureCode[0];
        }

        static TcpClient CreateTcpClient(string ipAddress, int port)
        {
            try
            {
                return new TcpClient(ipAddress, port);
            }
            catch (Exception ex)
            {
                WriteLine($"Error creating TCP client: {ex.Message}");
                throw; // Rethrow to handle it in the Main method
            }
        }

        static void ReadDiscreteInputs(ModbusIpMaster master)
        {
            WriteLine("Start reading inputs.");
            try
            {
                bool[] inputs = master.ReadInputs(coilInputStartAddress, numberOfDigitalInputs);
                DisplayDiscreteInputs(inputs);
            }
            catch (Exception ex)
            {
                WriteLine($"Error reading discrete inputs: {ex.Message}");
                _ = ReadKey(true);
            }
        }

        static void DisplayDiscreteInputs(bool[] inputs)
        {
            WriteLine("Discrete Inputs:");
            for (int i = 0; i < inputs.Length; i++)
            {
                WriteLine($"Input {coilInputStartAddress + i + 1}: {(inputs[i] ? "On" : "Off")}");
            }
        }
    }
}