﻿/*******************************************************************************************************************************************************
*    All Project Files Copyright © 2025 by The ep5 Educational Broadcasting Foundation                                                                 *
*                                                                                                                                                      *
*    Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:    *
*                                                                                                                                                      *
*        →  Redistributions of source code must retain the above copyright notice, this list of conditions, and the following disclaimer:              *
*        →  Redistributions in binary form must reproduce the above copyright notice, this list of conditions, and the following disclaimer in the     *
*           documentation and/or other materials provided with the distribution.                                                                       *
*        →  Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived from this    *
*           software without specific prior written permission.                                                                                        *
*                                                                                                                                                      *
*    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS “AS IS” AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED     *
*    TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO  EVENT SHALL THE COPYRIGHT HOLDER OR     *
*    CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,         *
*    PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF         *
*    LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT, INCLUDING NEGLIGENCE OR OTHERWISE, ARISING IN ANY WAY OUT OF THE USE OF THIS           *
*    SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                                                                      *
*******************************************************************************************************************************************************/

//  Written/updated 24 July 2025 by M Harb and David Fisher
//  Copyright © 2025 by The ep5 Educational Broadcasting Foundation; all rights reserved

using Modbus.Device;                                                 // Hard dependency on Nmodbus4
using System.Net.Sockets;
using static System.Console;

namespace ProtosXanalogueDemo
{
    internal class Program
    {
        private static readonly string ipAddress = "10.10.1.1";         // Bus coupler IP address
        private static readonly int port = 502;                         // Default Modbus TCP port
        private static readonly ushort TCregisterStartAddress = 17;     // Starting address for reading PX-332-J inputs
        private static readonly ushort RTDregisterStartAddress = 25;    // Starting address for reading PX-322-1 inputs
        private static readonly ushort TCnumberOfPoints = 8;            // Number of values to read
        private static readonly ushort RTDnumberOfPoints = 4;           // Number of values to read

        private static void Main(string[] args)
        {
            BackgroundColor = ConsoleColor.DarkBlue;
            ForegroundColor = ConsoleColor.White;
            Clear();
            using (TcpClient client = CreateTcpClient(ipAddress, port))
            {
                // Create client object and read the analogue inputs from the two PX-332-J terminals
                ModbusIpMaster modbusMaster = ModbusIpMaster.CreateIp(client);
                WriteLine("Created Modbus master.");
                ReadTCs(modbusMaster);
            }
        }

        private static TcpClient CreateTcpClient(string ipAddress, int port)
        {
            // Simple attempt at debugging network issues
            try
            {
                return new TcpClient(ipAddress, port);
            }
            catch (Exception ex)
            {
                WriteLine($"Error creating TCP client: {ex.Message}");
                throw; // Rethrow to handle it in the Main method
            }
        }

        private static void ReadTCs(ModbusIpMaster master)
        {
            while (true)
            {
                ushort[] TCvalues = master.ReadInputRegisters(TCregisterStartAddress, TCnumberOfPoints);
                Write("T/C value #1: " + (TCvalues[0] / 10.0).ToString("F1") + "°C  ");
                Write("T/C value #2: " + (TCvalues[2] / 10.0).ToString("F1") + "°C  ");
                Write("T/C value #3: " + (TCvalues[4] / 10.0).ToString("F1") + "°C  ");
                WriteLine("T/C value #4: " + (TCvalues[6] / 10.0).ToString("F1") + "°C");
                ushort[] RTDvalues = master.ReadInputRegisters(RTDregisterStartAddress, RTDnumberOfPoints);
                Write("RTD value #1: " + (RTDvalues[0] / 10.0).ToString("F1") + "°C  ");
                WriteLine("RTD value #2: " + (RTDvalues[2] / 10.0).ToString("F1") + "°C  ");
                Thread.Sleep(250);
            }
        }
    }
}