﻿/*******************************************************************************************************************************************************
*    All Project Files Copyright © 2025 by The ep5 Educational Broadcasting Foundation                                                                 *
*                                                                                                                                                      *
*    Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:    *
*                                                                                                                                                      *
*        →  Redistributions of source code must retain the above copyright notice, this list of conditions, and the following disclaimer:              *
*        →  Redistributions in binary form must reproduce the above copyright notice, this list of conditions, and the following disclaimer in the     *
*           documentation and/or other materials provided with the distribution.                                                                       *
*        →  Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived from this    *
*           software without specific prior written permission.                                                                                        *
*                                                                                                                                                      *
*    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS “AS IS” AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED     *
*    TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO  EVENT SHALL THE COPYRIGHT HOLDER OR     *
*    CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,         *
*    PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF         *
*    LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT, INCLUDING NEGLIGENCE OR OTHERWISE, ARISING IN ANY WAY OUT OF THE USE OF THIS           *
*    SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                                                                      *
*******************************************************************************************************************************************************/

//  Written 6 June 2025 by M Harb and David Fisher
//  Copyright © 2025 by The ep5 Educational Broadcasting Foundation; all rights reserved

//  This demo program connects to a Sealevel type 463E-OEM I/O processor https://www.sealevel.com/seai-o-463e-oem/
//  Note: this program cannot be started until at least five seconds after powering up the I/O processor
//  Note: if the program is interrupted and does not terminate normally, the power to the 463E must be cycled

//  The code, as written, assumes that the I/O processor is connected to three 24-point opto-rack boards, populated
//  with all outputs; future versions of this program will expand this to four racks, with some configured as inputs

using Modbus.Device;
using System.Net.Sockets;
using static System.Console;

namespace ModbusDemo
{
    internal class ModbusProgram
    {
        private const string    IPaddress = "10.10.10.110";     // Modbus device IP
        private const int       Port = 502;                     // Default Modbus TCP port
        private const byte      SlaveID = 247;                  // Modbus slave ID
        private const ushort    RegisterStartAddress = 0;       // Starting address to read
        private const ushort    NumRegisters = 9;               // Number of registers to read
        private const ushort    CoilStartAddress = 0;           // Starting address to write coils

        private static void Main(string[] args)
        {
            WriteLine("ModbusDemo begins here...\n\n");
            try
            {
                using TcpClient client = new(IPaddress, Port);
                // Build a client
                ModbusIpMaster master = ModbusIpMaster.CreateIp(client);
                WriteLine("CreateIp done\n");
                // Read register values
                ushort[] registers = master.ReadHoldingRegisters(SlaveID, RegisterStartAddress, NumRegisters);
                WriteLine("ReadHoldingRegisters done\n");
                DisplayRegisters(registers, RegisterStartAddress);
                WriteLine("\nDisplayRegisters done\n");
                // Set all channels as outputs
                master.WriteSingleRegister(SlaveID, 2, 0);
                WriteLine("WriteSingleRegister done.");
                // Write to specific outputs
                WriteIndividualCoils(master, registers, CoilStartAddress);
                WriteLine("WriteIndividualCoils done\n");
                // Write to groups of eight outputs at a time
                WriteAllOutputs(SlaveID, master);
                WriteLine("WriteAllOutputs done\n");
                ShutDown(SlaveID, master);
            }
            catch (Exception ex)
            {
                WriteLine("An error occurred:\n\n" + ex.Message);
            }
            finally
            {
                WriteLine("\nModbusDemo ends here.\n");
                _ = ReadKey(true);
            }
        }

        // Print registers we found and their values
        private static void DisplayRegisters(ushort[] registers, ushort startAddress)
        {
            WriteLine($"Read {registers.Length} registers starting at address {startAddress}:");
            for (int i = 0; i < registers.Length; i++)
            {
                WriteLine($"Register {startAddress + i}: {registers[i]}");
            }
        }

        // Exercise two racks, eight points at a time
        private static void WriteAllOutputs(ushort slaveID, ModbusIpMaster master)
        {
            bool[] valueArrayON = [true, false, true, false, true, false, true, false];
            bool[] valueArrayOFF = [false, true, false, true, false, true, false, true];

            WriteLine("Writing to three boards at once:\n");
            for (ushort j = 0; j < 4; j++)
            {
                for (ushort i = 0; i < 8; i++)
                {
                    master.WriteMultipleCoils((byte)slaveID, (ushort)(i * 8), valueArrayON);
                }
                Thread.Sleep(150);
                for (ushort i = 0; i < 8; i++)
                {
                    master.WriteMultipleCoils((byte)slaveID, (ushort)(i * 8), valueArrayOFF);
                }
            }
        }

        // Execise individual outputs
        private static void WriteIndividualCoils(ModbusIpMaster master, ushort[] registers, ushort coilStartAddress)
        {
            WriteLine("The value of 'coilStartAddress' is {0}", coilStartAddress);
            for (int i = 0; i < registers.Length; i++)
            {
                WriteLine(registers[i]);
            }
            DisplayRegisters(registers, RegisterStartAddress);
            for (ushort j = 0; j < 4; j++)
            {
                WriteLine("Turning the outputs ON.");

                for (ushort i = 0; i < 64; i++)
                {
                    master.WriteSingleCoil(SlaveID, i, true);
                }
                for (ushort i = 0; i < 64; i++)
                {
                    master.WriteSingleCoil(SlaveID, i, false);
                }
            }
            WriteLine("\nDone writing individual outputs!\n\n");
        }

        // Shut off all outputs
        private static void ShutDown(ushort slaveID, ModbusIpMaster master)
        {
            bool[] valueArrayOFF = new bool[8];

            for (ushort i = 0; i < 8; i++)
            {
                master.WriteMultipleCoils((byte)slaveID, (ushort)(i * 8), valueArrayOFF);
            }
        }
    }
}