﻿/*******************************************************************************************************************************************************
*    All Project Files Copyright © 2025 by The ep5 Educational Broadcasting Foundation                                                                 *
*                                                                                                                                                      *
*    Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:    *
*                                                                                                                                                      *
*        →  Redistributions of source code must retain the above copyright notice, this list of conditions, and the following disclaimer:              *
*        →  Redistributions in binary form must reproduce the above copyright notice, this list of conditions, and the following disclaimer in the     *
*           documentation and/or other materials provided with the distribution.                                                                       *
*        →  Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote products derived from this    *
*           software without specific prior written permission.                                                                                        *
*                                                                                                                                                      *
*    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS “AS IS” AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED     *
*    TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO  EVENT SHALL THE COPYRIGHT HOLDER OR     *
*    CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,         *
*    PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF         *
*    LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT, INCLUDING NEGLIGENCE OR OTHERWISE, ARISING IN ANY WAY OUT OF THE USE OF THIS           *
*    SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                                                                                      *
*******************************************************************************************************************************************************/

//  Written/updated 4 July 2025 by M Harb and David Fisher.
//  Copyright © 2025 by The ep5 Educational Broadcasting Foundation; all rights reserved.

//  This demo program connects to a Watlow F4SH-FAA0-01RG ramp-and-soak process controller;
//  https://www.watlow.com/products/controllers/temperature-and-process-controllers/series-f4-process-controller
//  Note: this program utilizes the NModbus API for access to the hardware.

//  The code, as written, assumes that the F4 analogue input 1 is connected to a 100 Ω platinum RTD
//  NB: This controller is obsolescent and does not reflect current process control state of the art. It is used as a convenience, as it was already
//  in the ep5 inventory. We would like to expand this module to other, more modern devices, providing that we can obtain samples for testing.

using Modbus.Device;            // NModbus namespace
using System.IO.Ports;
using static System.Console;

internal class Program
{
    // Registers
    private const ushort    ReadInput1 = 100;
    private const ushort    Input1error = 101;
    private const ushort    ALARM1STATUS = 102;
    private const ushort    ALARM2STATUS = 106;
    private const ushort    AnalogInput1decimal = 606;
    private const ushort    PowerOutput1A = 103;
    private const ushort    SetPoint1 = 300;
    private const ushort    CurrentDay = 1920;
    private const ushort    CurrentMonth = 1919;
    private const ushort    CurrentYear = 1921;
    private const ushort    CurrentHour = 1916;
    private const ushort    CurrentMinute = 1917;
    private const ushort    CurrentSecond = 1918;
    private const ushort    Output1Afunction = 700;
    private const ushort    Output1AcycleTimeType = 509;
    private const ushort    Output1AcycleTime = 506;
    private const ushort    DIGITALINPUT1FUNCTION = 1060;
    private const ushort    DIGITALINPUT1CONDITION = 1061;
    private const ushort    DIGITALINPUT1STATUS = 201;
    private const ushort    PowerOutAction = 1206;
    private const ushort    POWEROUTTIME = 1213;
    private const ushort    AnalogInputUnits = 608;
    private const ushort    AnalogInput1AsensorType = 601;
    private const ushort    AnalogInput1sensor = 600;
    private const ushort    AnalogInput1setPointHighLimit = 603;
    private const ushort    AnalogInput1setPointLowLimit = 602;
    private const ushort    TempScaleDisplay = 1923;
    private const ushort    TempScale = 901;
    private const ushort    AnalogOutput1Atype = 701;
    private const ushort    HighPowerLimitControlOutput1A = 714;
    private const ushort    ProportionalBand = 500;

    // Configuration
    private const byte      slaveID = 1;
    private const ushort    DeadBottom = 400;

    private static void Main(string[] args)
    {
        ushort[] registers;

        // Initialize the F4
        SerialPort serialPort = (SerialPort)InitializeModbusSerialPort();
        ModbusSerialMaster master = (ModbusSerialMaster)InitializeModbusMaster(serialPort);

        // Then, run it
        try
        {
            // Monitor continuously
            while (true)
            {
                // Check for escape key press
                if (KeyAvailable)
                {
                    ConsoleKeyInfo key = ReadKey(true);
                    if (key.Key == ConsoleKey.Escape)
                    {
                        WriteLine("<ESC> pressed. Exiting...");
                        break;
                    }
                }

                // Display current process values
                try
                {
                    registers = master.ReadInputRegisters(slaveID, Input1error, 1);
                    if (registers[0] != 0)
                    {
                        WriteLine("Failure in the temperature input channel; program ends here...\n");
                        return;
                    };
                    registers = master.ReadInputRegisters(slaveID, ReadInput1, 1);
                    Write("temperature is {0:F1}; ", registers[0] / 10.0);
                    registers = master.ReadInputRegisters(slaveID, PowerOutput1A, 1);
                    Write("power output is {0:F2}%\n", registers[0] / 100.0);
                    // Pause before next read
                    Thread.Sleep(500);
                }
                catch (Exception ex)
                {
                    WriteLine($"Error: {ex.Message}");
                    break;
                }
            }
        }
        catch (Exception ex)
        {
            WriteLine($"Initialization error: {ex.Message}");
        }
        finally
        {
            // Shut off the output by changing setpoint to minimum allowed value
            // This assumes that the current process value will be greater than the minimum allowable setpoint
            // This weakness appears to be driven by the limitations of the F4 controller itself
            master?.WriteSingleRegister(slaveID, SetPoint1, DeadBottom);

            // Clean up
            master?.Dispose();
            serialPort?.Close();
        }

        static object InitializeModbusSerialPort()
        {
            SerialPort serialPort;

            serialPort = new SerialPort
            {
                PortName = "COM3",
                BaudRate = 19200,
                DataBits = 8,
                Parity = Parity.None,
                StopBits = StopBits.One,
                ReadTimeout = 1000
            };
            serialPort.Open();
            return serialPort;
        }

        static object InitializeModbusMaster(SerialPort serialPort)
        {
            ushort[] registers;
            ushort setPoint = 770;
            ushort analogueSetpointMaximum = 1250;
            ushort analogueOutputType = 4;   // 0 = 4-20ma; 4 = 0-10vdc
            ModbusSerialMaster master;
            DateTime dateTime;

            master = ModbusSerialMaster.CreateRtu(serialPort);

            // Set sensor type
            WriteLine("Set sensor type to 100 Ω DIN platinum RTD");
            master.WriteSingleRegister(slaveID, AnalogInput1sensor, 1);
            master.WriteSingleRegister(slaveID, AnalogInput1AsensorType, 11);

            // Set to one decimal place (this cannot be the first initialization instruction)
            WriteLine("Set to one decimal place on display");
            master.WriteSingleRegister(slaveID, AnalogInput1decimal, 1);

            // Set low and high limit values for analogue1 setpoint
            master.WriteSingleRegister(slaveID, AnalogInput1setPointLowLimit, DeadBottom);
            master.WriteSingleRegister(slaveID, AnalogInput1setPointHighLimit, analogueSetpointMaximum);

            // Display current setpoint high and low limits
            registers = master.ReadInputRegisters(slaveID, AnalogInput1setPointLowLimit, 1);
            WriteLine("Setpoint low limit is {0}", registers[0]);
            registers = master.ReadInputRegisters(slaveID, AnalogInput1setPointHighLimit, 1);
            WriteLine("Setpoint high limit is {0}", registers[0]);

            // Display analogue output high limit
            registers = master.ReadInputRegisters(slaveID, HighPowerLimitControlOutput1A, 1);
            WriteLine("Analogue output high limit is {0}%", registers[0]);

            // Set to proportional mode
            WriteLine("Make proportional band 5°F");
            master.WriteSingleRegister(slaveID, ProportionalBand, 50);

            // Set output function to heating
            WriteLine("Output function is heating");
            master.WriteSingleRegister(slaveID, Output1Afunction, 1);

            // Set analog output1 to appropriate type
            WriteLine("Set the analog output to correct type");
            master.WriteSingleRegister(slaveID, AnalogOutput1Atype, analogueOutputType);

            // Set analog input parameter to temperature
            master.WriteSingleRegister(slaveID, AnalogInputUnits, 0);

            // Set temperature SCALE to ON
            master.WriteSingleRegister(slaveID, TempScaleDisplay, 1);

            // Set temperature scale type to Fahrenheit
            master.WriteSingleRegister(slaveID, TempScale, 0);

            // Set power failure response
            master.WriteSingleRegister(slaveID, PowerOutAction, 2);

            // Set output1 cycle time type to variable burst
            master.WriteSingleRegister(slaveID, Output1AcycleTimeType, 0);

            // Read current output1 cycle time
            registers = master.ReadInputRegisters(slaveID, Output1AcycleTime, 1);
            WriteLine("Output cycle time is {0}", registers[0]);

            // Set output1 cycle time to new value
            master.WriteSingleRegister(slaveID, Output1AcycleTime, 500);

            // Check new value
            registers = master.ReadInputRegisters(slaveID, Output1AcycleTime, 1);
            WriteLine("Output cycle time is now {0}", registers[0]);

            // Write setpoint to F4
            WriteLine("Setting the process variable setpoint");
            master.WriteSingleRegister(slaveID, SetPoint1, setPoint);

            // Verify setpoint value
            WriteLine("Verifying the setpoint value");
            registers = master.ReadInputRegisters(slaveID, SetPoint1, 1);
            WriteLine("Setpoint is {0:F1}", registers[0] / 10.0);

            // Set F4 clock to current date and time
            dateTime = DateTime.Now;
            master.WriteSingleRegister(slaveID, CurrentDay, (ushort)dateTime.Day);
            master.WriteSingleRegister(slaveID, CurrentMonth, (ushort)dateTime.Month);
            master.WriteSingleRegister(slaveID, CurrentYear, (ushort)dateTime.Year);
            master.WriteSingleRegister(slaveID, CurrentHour, (ushort)dateTime.Hour);
            master.WriteSingleRegister(slaveID, CurrentMinute, (ushort)dateTime.Minute);
            master.WriteSingleRegister(slaveID, CurrentSecond, (ushort)dateTime.Second);
            WriteLine("Set F4 clock to {0} {1}", dateTime.ToLongTimeString(), dateTime.ToLongDateString());

            return master;
        }
    }
}
